/*
 * Copyright 2025 Raphael Mudge, Adversary Fan Fiction Writers Guild
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of
 * conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other materials provided
 * with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS “AS IS” AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <windows.h>

typedef struct {
	__typeof__(LoadLibraryA)   * LoadLibraryA;
	__typeof__(GetProcAddress) * GetProcAddress;
} IMPORTFUNCS;

WINUSERAPI int WINAPI USER32$MessageBoxA(HWND hWnd,LPCSTR lpText,LPCSTR lpCaption,UINT uType);

/*
 * our xorkey, we're going to set this via our loader.spec
 */
char xorkey[128] = { 1 };

/*
 * A simple routine to obfuscate and de-obfuscate our data
 */
void applyxor(char * data, DWORD len) {
	for (DWORD x = 0; x < len; x++) {
		data[x] ^= xorkey[x % 128];
	}
}

/* globals to keep track of our DLL in memory. For simplicity's sake, this example
 * assumes the whole thing is RWX, but we could really do whatever we need between
 * the loader and this hooking module */
char * g_dllBase;
DWORD  g_dllSize;

/*
 * Our hooked functions that do what we want...
 */
int WINAPI _MessageBoxA(HWND hWnd,LPCSTR lpText,LPCSTR lpCaption,UINT uType) {
	int result;

	applyxor(g_dllBase, g_dllSize);

	// may as well use our own strings, because the originals are garbled right now
	result = USER32$MessageBoxA(hWnd, "Hello from hook.c!", "HOOKED!", uType);

	applyxor(g_dllBase, g_dllSize);

	return result;
}

/*
 * This is the GetProcAddress our loader will use. It's a chance for us to hook things. Notice, we also use
 * this to replace any other references to GetProcAddress with our hook function too. In this way, for the
 * things our DLL does... everything will use this GetProcAddress giving us nice visibility into everything.
 */
char * WINAPI _GetProcAddress(HMODULE hModule, LPCSTR lpProcName) {
	char * result = (char *)GetProcAddress(hModule, lpProcName);

	if ((char *)GetProcAddress == result) {
		return (char *)_GetProcAddress;
	}
	/* I like this method of identifying hook targets, because there are no strings to fuss with. BUT,
	 * beware... LIBRARY$func WILL LoadLibrary the specified library, here USER32.dll when the PICO is
	 * loaded into memory. So, if that's not desireable, you may need to switch to using string hashes or
	 * something else to decide what to hook */
	else if ((char *)USER32$MessageBoxA == result) {
		return (char *)_MessageBoxA;
	}

	return result;
}

/* Our entry point and our initialization for everything fun that will happen... */
void go(IMPORTFUNCS * funcs, char * dllBase, DWORD dllsz) {
	/* update the GetProcAddress our loader uses */
	funcs->GetProcAddress = (__typeof__(GetProcAddress) *)_GetProcAddress;

	/* track this information, because we will need it later */
	g_dllBase = dllBase;
	g_dllSize = dllsz;
}
