package crystalpalace.spec;

import java.io.*;

/**
 * An output message generated by Crystal Palace
 */
public class SpecMessage {
	/** echo message */
	public static final int MESSAGE_ECHO = 0;

	/** the source of the message */
	protected SpecProgram source;

	/** string contents of the output */
	protected String message;

	/** type of the message */
	protected int type;

	/**
	 * Instantiate a new message
	 *
	 * @param source the program this message came from
	 * @param message the contents of the output
	 * @param type the type of message (e.g., @{link #MESSAGE_ECHO})
	 */
	protected SpecMessage(SpecProgram source, String message, int type) {
		this.source  = source;
		this.message = message;
		this.type    = type;
	}

	/**
	 * Get the contents of this log message
	 *
	 * @return the contents of our log message
	 */
	public String getMessage() {
		return message;
	}

	/**
	 * Get the type of this log message
	 *
	 * @return the message type (e.g., @{link #MESSAGE_ECHO})
	 */
	public int getType() {
		return type;
	}

	/**
	 * Get the label context from our current Spec program (e.g., x86, x64)
	 *
	 * @return the label target string
	 */
	public String getTarget() {
		return source.getLastTarget();
	}

	/**
	 * Get the shortname of the file where this message occured. In the case of {@code run "file.spec"}--the file associated with this message
	 * may differ from the top-level .spec file that was run via {@link LinkSpec}
	 *
	 * @return the short file name associated where this error occured.
	 */
	public String getFileName() {
		return new java.io.File(source.getFile()).getName();
	}

	/**
	 * Log a message for the echo command
	 *
	 * @param source the source program
	 * @param message the message contents
	 *
	 * @return an object containing this message
	 */
	public static SpecMessage Echo(SpecProgram source, String message) {
		return new SpecMessage(source, message, MESSAGE_ECHO);
	}

	/**
	 * Format our message into a pleasing enough string representation
	 *
	 * @return string representation of this message
	 */
	public String toString() {
		StringBuffer temp = new StringBuffer();
		temp.append(getMessage() + " in " + getFileName() + " ("+getTarget()+")");
		return temp.toString();
	}
}
