package crystalpalace.spec;

import crystalpalace.coff.*;
import crystalpalace.util.*;
import crystalpalace.export.*;

import java.util.*;
import java.io.*;

/**
 * An object that parses and interprets a Crystal Palace <a href="https://tradecraftgarden.org/specfiles.html">specification file</a>.
 * <p>
 *
 * Call {@link Capability#Parse} (ala {@code link}) or {@link Capability#None} (ala {@code piclink}) to create the capability argument.
 * <p>
 *
 * Call {@link #Parse(java.lang.String)} to create a LinkSpec object.
 * <p>
 *
 * Call {@link #run(Capability, Map)} to apply a specification file to a Capability and other $ARGUMENTS / %variables passed into the environment.
 * <p>
 *
 * That's the API. If there's a problem, the API will throw a {@link SpecParseException} or a {@link SpecProgramException}.
 * <p>
 *
 * This class is not thread safe.
 * <p>
 *
 * <strong>PERFORMANCE NOTE:</strong> The first use of this API, running a .spec file with a binary transform (e.g., code mutation,
 * link-time optimization, etc.) will incur a 500ms penalty. This is because the iced library has to load several constants from within
 * our JAR file. Future calls to this API (with or without binary transforms) should resolve quickly (~20ms in my VM environment).
 *
 * @author Raphael Mudge
 */
public class LinkSpec {
	/**
	 * the file associated with this spec.
	 */
	protected String parent     = ".";

	/**
	 * the author of this .spec file (if specified)
	 */
	protected String author     = "";

	/**
	 * the short name of the technique/tradecraft in this .spec file
	 */
	protected String name        = "";

	/**
	 * a description of the tradecraft/technique in this .spec file
	 */
	protected String description = "";

	/**
	 * the execution environment for applying this specification to passed in variables
	 */
	protected SpecProgram program;

	/**
	 * register a logger to receive output (e.g., echo command) from Crystal Palace.
	 *
	 * @param logger the object that receives messages from Crystal Palace.
	 */
	public void addLogger(SpecLogger logger) {
		program.addLogger(logger);
	}

	/**
	 * remove a registered logger
	 *
	 * @param logger the object that (should no longer) receive messages from Crystal Palace.
	 */
	public void removeLogger(SpecLogger logger) {
		program.removeLogger(logger);
	}

	/**
	 * Get the name of the file associated with this .spec. The files does not actually have to exist. It's set by the API when
	 * {@link #Parse} is called.
	 *
	 * @return the file associated with this spec.
	 */
	public String getFile() {
		return parent;
	}

	/**
	 * Return the author of this tradecraft. This value is set with the {@code author "name"} command in the .spec file.
	 *
	 * @return the author value
	 */
	public String getAuthor() {
		return author;
	}

	/**
	 * Return the name of the technique/tradecraft. This value is set with the {@code name "tradecraft name"} command in the .spec file.
	 *
	 * @return the name value
	 */
	public String getName() {
		return name;
	}

	/**
	 * Return a description of the technique/tradecraft. This value is set with the {@code describe "my technique does this"} command in the .spec file.
	 *
	 * @return the description value
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * Check if this specification has this specific target. Valid values include {@code x86, x64}.
	 *
	 * @param arch the target architecture to check
	 *
	 * @return true if the technique has an option to target the specific arch
	 */
	public boolean targets(String arch) {
		return program.targets(arch);
	}

	/**
	 * The constructor for our LinkSpec. Call {@link #Parse} to create an instance of this class.
	 *
	 * @param parser the .spec file parser created by {@link #Parse}
	 * @param parent the parent file for this .spec file, returned by {@link #getFile}
	 */
	protected LinkSpec(SpecParser parser, String parent) {
		this.parent = parent;

		/* make the default name of the spec, the filename only */
		name = new java.io.File(parent).getName();

		/* try to make this less generic if we can */
		if ("loader.spec".equals(name))
			name = new java.io.File(parent).getParentFile().getName();

		/* setup our interpreter, plz */
		program = new SpecProgram(parent);
	}

	/**
	 * Apply this specification file's directives to the passed in parameters.
	 *
	 * @param capability is our Win32 DLL or COFF argument. If it's a DLL, {@link #runDll} handles the rest. If it's a COFF, {@link #runObject}
	 *	handles the rest.
	 * @param vars is a mapping of {@code $KEY} to {@code byte[]} and {@code %VAR} to {@code "string"} values used in the specification file.
	 *	Each variable name must begin with the right sigil ($, %) to be accessible within the spec environment.
	 *
	 * @return the position-independent code (usually) generated by applying this specification to the arguments
	 *
	 * @throws SpecParseException if a specification file called {@code run "file.spec"} from this specification has syntax errors.
	 * @throws SpecProgramException if an error occurs or is detected during the application of this specification file.
	 *
	 * @deprecated as of release 12.01.25. Use {@link Capability#Parse} and {@link #run(Capability, Map)}
	 */
	@Deprecated
	public byte[] run(byte[] capability, Map vars) throws SpecParseException, SpecProgramException {
		return run(Capability.Parse(capability), vars);
	}

	/**
	 * Apply this specification file's directives to the passed in parameters.
	 *
	 * @param capability an object with our capability's content and the other capability-specific arguments needed for our program.
	 * @param vars is a mapping of {@code $KEY} to {@code byte[]} and {@code %VAR} to {@code "string"} values used in the specification file.
	 *	Each variable name must begin with the right sigil ($, %) to be accessible within the spec environment.
	 *
	 * @return the position-independent code (usually) generated by applying this specification to the arguments
	 *
	 * @throws SpecParseException if a specification file called {@code run "file.spec"} from this specification has syntax errors.
	 * @throws SpecProgramException if an error occurs or is detected during the application of this specification file.
	 */
	public byte[] run(Capability capability, Map vars) throws SpecParseException, SpecProgramException {
		/* populate our environment map */
		HashMap env = new HashMap();
		if (vars != null)
			env.putAll(vars);

		/* put our capability information into the environment (only if we have such info) */
		if (capability.hasCapability())
			env.put(capability.getKey(), capability.getContents());

		/* reset all of the error/state information */
		program.reset();

		/* apply this .spec to our capability */
		return program.run(capability.getLabel(), capability.getArch(), env);
	}

	/**
	 * Apply this specification file's directives to the passed in vars to set variables. The run specification file must end with
	 * an empty stack.
	 *
	 * @param capability an object with our capability's content and the other capability-specific arguments needed for our program.
	 * @param vars is a mapping of {@code $KEY} to {@code byte[]} and {@code %VAR} to {@code "string"} values used in the specification file.
	 *	Each variable name must begin with the right sigil ($, %) to be accessible within the spec environment.
	 *
	 * @return the updated vars object.
	 *
	 * @throws SpecParseException if a specification file called {@code run "file.spec"} from this specification has syntax errors.
	 * @throws SpecProgramException if an error occurs or is detected during the application of this specification file.
	 */
	public Map runConfig(Capability capability, Map vars) throws SpecParseException, SpecProgramException {
		/* put our capability information into the environment (only if we have such info) */
		if (capability.hasCapability())
			vars.put(capability.getKey(), capability.getContents());

		/* reset the program */
		program.reset();

		/* run our config */
		program.runConfig(capability.getLabel(), capability.getArch(), vars);

		/* return our original vars */
		return vars;
	}

	/**
	 * Apply this specification file's directives to the passed in parameters.
	 *
	 * @param dll_arg our Win32 DLL argument, installed into the environment as {@code $DLL}. This value is lightly checked for validity
	 * 	(e.g., MZ/PE headers exist). The architecture of this DLL determines which target to call from the specification file.
	 * @param vars is a mapping of {@code $KEY} to {@code byte[]} and {@code %VAR} to {@code "string"} values used in the specification file.
	 *	Each variable name must begin with the right sigil ($, %) to be accessible within the spec environment.
	 *
	 * @return the position-independent code (usually) generated by applying this specification to the arguments
	 *
	 * @throws SpecParseException if a specification file called {@code run "file.spec"} from this specification has syntax errors.
	 * @throws SpecProgramException if an error occurs or is detected during the application of this specification file.
	 *
	 * @deprecated as of release 12.01.25. Use {@link Capability#ParseDll} and {@link #run(Capability, Map)}
	 */
	@Deprecated
	public byte[] runDll(byte[] dll_arg, Map vars) throws SpecParseException, SpecProgramException {
		return run(Capability.ParseDll(dll_arg), vars);
	}

	/**
	 * Apply this specification file's directives to the passed in parameters.
	 *
	 * @param object_arg our Win32 COFF argument, installed into the environment as {@code $OBJECT}. This value is lightly checked for validity
	 * 	(e.g., valid Machine header value). The architecture of this object determines which target (x86.o, x64.o) to call from the specification file.
	 * @param vars a mapping of {@code $KEY} to {@code byte[]} values made available during the application of the specification file.
	 * 	Each key should begin with '$' to be accessible within the spec environment.
	 *
	 * @return the position-independent code (usually) generated by applying this specification to the arguments
	 *
	 * @throws SpecParseException if a specification file called {@code run "file.spec"} from this specification has syntax errors.
	 * @throws SpecProgramException if an error occurs or is detected during the application of this specification file.
	 *
	 * @deprecated as of release 12.01.25. Use {@link Capability#ParseObject} and {@link #run(Capability, Map)}
	 */
	@Deprecated
	public byte[] runObject(byte[] object_arg, Map vars) throws SpecParseException, SpecProgramException {
		return run(Capability.ParseObject(object_arg), vars);
	}

	/**
	 * Act on this specification file's directives.
	 *
	 * This API differs in purpose from {@link run}, {@link runDll}, and {@link runObject}. Its purpose is to assemble a PIC program
	 * from known parts, rather than applying PIC to a user-specified capability implemented as a DLL or COFF.
	 *
	 * @param arch is the target architecture label to act on (e.g., x86 or x64)
	 * @param vars a mapping of {@code $KEY} to {@code byte[]} values made available during the application of the specification file.
	 * 	Each key should begin with '$' to be accessible within the spec environment.
	 *
	 * @return the position-independent code generated by executing this specification file.
	 *
	 * @throws SpecParseException if a specification file called {@code run "file.spec"} from this specification has syntax errors.
	 * @throws SpecProgramException if an error occurs or is detected during the application of this specification file.
	 *
	 * @deprecated as of release 12.01.25. Use {@link Capability#None} and {@link #run(Capability, Map)}
	 */
	@Deprecated
	public byte[] buildPic(String arch, Map vars) throws SpecParseException, SpecProgramException {
		return run(Capability.None(arch), vars);
	}

	/**
	 * Parse a .spec file and return a ready-to-run {@code LinkSpec} object.
	 *
	 * @param parent the parent file of this .spec content. Returned by {@code #getFile}.
	 * @param content the .spec file content
	 *
	 * @return a LinkSpec object.
	 *
	 * @throws SpecParseException if a syntax error or mis-used command is detected
	 */
	public static LinkSpec Parse(String parent, String content) throws SpecParseException {
		SpecParser parser = new SpecParser();
		parser.parse(content, parent);

		if (parser.getErrors().size() > 0) {
			throw new SpecParseException(parser, parent);
		}

		return parser.getSpec();
	}

	/**
	 * Parse a .spec file and return a ready-to-run {@code LinkSpec} object.
	 *
	 * @param parent The name of the .spec file to read and parse.
	 *
	 * @return a LinkSpec object
	 *
	 * @throws SpecParseException if a syntax error or mis-used command is detected
	 * @throws IOException if we can't read the .spec file
	 */
	public static LinkSpec Parse(String parent) throws SpecParseException, IOException {
		return Parse( parent, CrystalUtils.readStringFromFile(parent) );
	}
}
