package crystalpalace.spec;

import crystalpalace.coff.*;
import crystalpalace.util.*;
import crystalpalace.export.*;

import java.util.*;
import java.io.*;

public class SpecParser {
	protected static Set  commands   = CrystalUtils.toSet("export, generate, patch, preplen, prepsum, link, load, make, push, xor, rc4, run, import, disassemble, coffparse, merge, reladdr, dfr, fixptrs, mergelib, fixbss, remap");
	protected static Set  commandsv  = CrystalUtils.toSet("push, xor, rc4");
	protected static Set  fullcmds   = CrystalUtils.toSet("make coff, make object, make pic, make pic64, export, preplen, prepsum, merge");
	protected static Set  arg1cmds   = CrystalUtils.toSet("import, link, load, run, disassemble, coffparse, reladdr, fixptrs, fixbss, mergelib");
	protected static Set  labels     = CrystalUtils.toSet("x86, x64, x86.o, x64.o, x86.dll, x64.dll");

	protected static Set  options    = CrystalUtils.toSet("+optimize,+disco,+mutate,+gofirst");
	protected static Set  optcmds    = CrystalUtils.toSet("make object, make pic, make pic64, make coff");

	protected LinkSpec spec   = null;
	protected List     errors = new LinkedList();
	protected String   parent = "";

	public LinkSpec getSpec() {
		return spec;
	}

	public List getErrors() {
		return errors;
	}

	public void error(String msg, int lineNo) {
		errors.add(msg + " at line " + lineNo);
	}

	/* simplify our number parser and error reporting logic */
	public int parseInt(CommandParser parser, String arg, int x) {
		int val = CrystalUtils.parseInt(arg, -1);

		if (val == -1)
			error("Invalid argument for '" + parser.getOriginal() + "'. " + arg + " is not a valid number", (x + 1));
		else if (val < 0)
			error("Invalid argument for '" + parser.getOriginal() + "'. " + arg + " is a negative number", (x + 1));
		else if (val == 0)
			error("Invalid argument for '" + parser.getOriginal() + "'. " + arg + " is zero", (x + 1));

		return val;
	}

	public void parse(String content, String parent) {
                         spec     = new LinkSpec(this, parent);
		String   label    = "";

		String contents[] = content.split("\\n");
		for (int x = 0; x < contents.length; x++) {
			/* handle comments */
			if (contents[x].indexOf("#") >= 0) {
				contents[x] = contents[x].substring(0, contents[x].indexOf("#"));
			}

			/* trim out any whitespace */
			contents[x] = contents[x].trim();

			/* carry on! */
			if (contents[x].endsWith(":") && contents[x].indexOf(' ') == -1) {
				/* it's a label */
				label = contents[x].substring(0, contents[x].length() - 1);

				/* check that it's a valid label! */
				if (!labels.contains(label))
					error("Invalid label '" + label + "' - acceptable values are x86, x64", x + 1);

				continue;
			}
			else if ("".equals(contents[x])) {
				/* it's whitespace, do nothing */
				continue;
			}

			CommandParser parser  = new CommandParser(contents[x]);
			String        command = parser.getCommand();
			String        args[]  = parser.getArguments();

			if ("describe".equals(command)) {
				if (parser.getArguments().length == 0)
					error("Command '" + command + "' requires an argument", x + 1);
				else
					spec.description = parser.getArguments()[0];

			}
			else if ("author".equals(command)) {
				if (parser.getArguments().length == 0)
					error("Command '" + command + "' requires an argument", x + 1);
				else
					spec.author = parser.getArguments()[0];
			}
			else if ("name".equals(command)) {
				if (parser.getArguments().length == 0)
					error("Command '" + command + "' requires an argument", x + 1);
				else
					spec.name = parser.getArguments()[0];
			}
			else if ("".equals(label)) {
				/* no top-level label? That's a (fatal) error! */
				error("Commands must exist under an 'x86:' or 'x64:' label", (x + 1));
				return;
			}
			else {
				int           alen    = parser.getArguments().length;

				/* if we don't recognize the command, bail */
				if (!commands.contains(command)) {
					error("Invalid command '" + command + "'", (x + 1));
					continue;
				}

				/* check if we're correct, first */
				if (fullcmds.contains(contents[x])) {
					//parser.setOption("+mutate");
					spec.program.getInstructionsForLabel(label).add(parser);
					continue;
				}
				else if (parser.hasOptions()) {
					/* check that our options are valid */
					Iterator i = parser.getOptions().iterator();
					while (i.hasNext()) {
						String opt = (String)i.next();
						if (!options.contains(opt))
							error("Invalid option " + opt + " for '" + parser.getFullCommand() + "'", (x + 1));
					}

					/* check that the command accepts options */
					if ( optcmds.contains(parser.getFullCommand()) ) {
						//parser.setOption("+mutate");
						spec.program.getInstructionsForLabel(label).add(parser);
					}
					else {
						error("Command '" + command + "' does not accept +options " + parser.getOptions(), (x + 1));
					}

					continue;
				}
				else if (commandsv.contains(command) && alen == 1) {
					if (args[0].startsWith("$")) {
						spec.program.getInstructionsForLabel(label).add(parser);
					}
					else {
						error("Invalid argument for '" + parser.getOriginal() + "'. Try $" + args[0], (x + 1));
					}
					continue;
				}
				else if (arg1cmds.contains(command) && alen == 1) {
					spec.program.getInstructionsForLabel(label).add(parser);
					continue;
				}
				else if ("dfr".equals(command) && (alen == 2 || alen == 3)) {
					Set valid = CrystalUtils.toSet("ror13, strings");

					if (valid.contains(args[1])) {
						spec.program.getInstructionsForLabel(label).add(parser);
					}
					else {
						error("Invalid method '" + args[1] + "' for '" + parser.getOriginal() + "'. Use 'ror13' or 'strings'", (x + 1));
					}

					continue;
				}
				else if ("remap".equals(command) && alen == 2) {
					spec.program.getInstructionsForLabel(label).add(parser);
					continue;
				}
				else if ("load".equals(command) && alen == 2) {
					if (!args[0].startsWith("$")) {
						error("Invalid argument for '" + parser.getOriginal() + "'. Try $" + args[0], (x + 1));
					}
					else {
						spec.program.getInstructionsForLabel(label).add(parser);
					}
					continue;
				}
				else if ("generate".equals(command) && alen == 2) {
					int val = parseInt(parser, args[1], x);

					if (!args[0].startsWith("$")) {
						error("Invalid argument for '" + parser.getOriginal() + "'. Try $" + args[0], (x + 1));
					}
					else if (val != -1) {
						spec.program.getInstructionsForLabel(label).add(parser);
					}

					continue;
				}
				else if ("patch".equals(command) && alen == 2) {
					if (!args[1].startsWith("$")) {
						error("Invalid argument for '" + parser.getOriginal() + "'. Try $" + args[1], (x + 1));
					}
					else {
						spec.program.getInstructionsForLabel(label).add(parser);
					}

					continue;
				}

				/* start preparing our error message about the command, start with correct use hint */
				String hint = "";
				if ("export".equals(command)) {
					hint = "export";
				}
				else if ("generate".equals(command)) {
					hint = "generate $KEY 1024";
				}
				else if ("link".equals(command)) {
					hint = "link 'section_name'";
				}
				else if ("load".equals(command)) {
					hint = "load 'path/to/file'";
				}
				else if ("run".equals(command)) {
					hint = "run 'path/to/file'";
				}
				else if ("mergelib".equals(command)) {
					hint = "mergelib 'path/to/file.zip'";
				}
				else if ("disassemble".equals(command)) {
					hint = "disassemble out.txt";
				}
				else if ("coffparse".equals(command)) {
					hint = "coffparse out.txt";
				}
				else if ("make".equals(command)) {
					hint = "make pic, make coff, make object";
				}
				else if ("push".equals(command)) {
					hint = "push $DLL";
				}
				else if ("xor".equals(command)) {
					hint = "xor $KEY";
				}
				else if ("rc4".equals(command)) {
					hint = "rc4 $KEY";
				}
				else if ("patch".equals(command)) {
					hint = "patch 'symbol' $VAR";
				}
				else if ("import".equals(command)) {
					hint = "import 'LoadLibraryA, GetProcAddress, ...'";
				}
				else if ("reladdr".equals(command)) {
					hint = "reladdr '_go'";
				}
				else if ("dfr".equals(command)) {
					hint = "dfr 'resolver_func' 'ror13|strings' ['mod1, mod2']";
				}
				else if ("fixptrs".equals(command)) {
					hint = "fixptrs '_getretaddr'";
				}
				else if ("fixbss".equals(command)) {
					hint = "fixbss 'getbssaddr'";
				}
				else if ("remap".equals(command)) {
					hint = "remap 'old_symbol' 'new_symbol'";
				}


				/* report the error */
				if (alen == 0) {
					error("Command " + command + " missing arguments, try '" + hint + "'", (x + 1));
				}
				else {
					error("Command " + command + " invalid arguments, try '" + hint + "'", (x + 1));
				}
			}
		}
	}
}
