package crystalpalace.spec;

import crystalpalace.coff.*;
import crystalpalace.util.*;
import crystalpalace.export.*;

import java.util.*;
import java.io.*;

/**
 * An object that parses and interprets a Crystal Palace <a href="https://tradecraftgarden.org/specfiles.html">specification file</a>.
 * <p>
 *
 * Call {@link #Parse} to create a LinkSpec object.
 * <p>
 *
 * Call {@link #run} to apply a specification file to a DLL (or COFF) and other $ARGUMENTS passed into the environment.
 * <p>
 *
 * That's the API. If there's a problem, the API will throw a {@link SpecParseException} or a {@link SpecProgramException}.
 * <p>
 *
 * This class is not thread safe.
 * <p>
 *
 * <strong>PERFORMANCE NOTE:</strong> The first use of this API, running a .spec file with a binary transform (e.g., code mutation,
 * link-time optimization, etc.) will incur a 500ms penalty. This is because the iced library has to load several constants from within
 * our JAR file. Future calls to this API (with or without binary transforms) should resolve quickly (~20ms in my VM environment).
 *
 * @author Raphael Mudge
 */
public class LinkSpec {
	/**
	 * the file associated with this spec.
	 */
	protected String parent     = ".";

	/**
	 * the author of this .spec file (if specified)
	 */
	protected String author     = "";

	/**
	 * the short name of the technique/tradecraft in this .spec file
	 */
	protected String name        = "";

	/**
	 * a description of the tradecraft/technique in this .spec file
	 */
	protected String description = "";

	/**
	 * the execution environment for applying this specification to passed in variables
	 */
	protected SpecProgram program;

	/**
	 * Get the name of the file associated with this .spec. The files does not actually have to exist. It's set by the API when
	 * {@link #Parse} is called.
	 *
	 * @return the file associated with this spec.
	 */
	public String getFile() {
		return parent;
	}

	/**
	 * Return the author of this tradecraft. This value is set with the {@code author "name"} command in the .spec file.
	 *
	 * @return the author value
	 */
	public String getAuthor() {
		return author;
	}

	/**
	 * Return the name of the technique/tradecraft. This value is set with the {@code name "tradecraft name"} command in the .spec file.
	 *
	 * @return the name value
	 */
	public String getName() {
		return name;
	}

	/**
	 * Return a description of the technique/tradecraft. This value is set with the {@code describe "my technique does this"} command in the .spec file.
	 *
	 * @return the description value
	 */
	public String getDescription() {

		return description;
	}

	/**
	 * Check if this specification has this specific target. Valid values include {@code x86, x64}.
	 *
	 * @param arch the target architecture to check
	 *
	 * @return true if the technique has an option to target the specific arch
	 */
	public boolean targets(String arch) {
		return program.targets(arch);
	}

	/**
	 * The constructor for our LinkSpec. Call {@link #Parse} to create an instance of this class.
	 *
	 * @param parser the .spec file parser created by {@link #Parse}
	 * @param parent the parent file for this .spec file, returned by {@link #getFile}
	 */
	protected LinkSpec(SpecParser parser, String parent) {
		this.parent = parent;

		/* make the default name of the spec, the filename only */
		name = new java.io.File(parent).getName();

		/* try to make this less generic if we can */
		if ("loader.spec".equals(name))
			name = new java.io.File(parent).getParentFile().getName();

		/* setup our interpreter, plz */
		program = new SpecProgram(parent);
	}

	/**
	 * Apply this specification file's directives to the passed in parameters.
	 *
	 * @param capability is our Win32 DLL or COFF argument. If it's a DLL, {@link #runDll} handles the rest. If it's a COFF, {@link #runObject}
	 *	handles the rest.
	 * @param vars a mapping of {@code $KEY} to {@code byte[]} values made available during the application of the specification file.
	 * 	Each key should begin with '$' to be accessible within the spec environment.
	 *
	 * @return the position-independent code (usually) generated by applying this specification to the arguments
	 *
	 * @throws SpecParseException if a specification file called {@code run "file.spec"} from this specification has syntax errors.
	 * @throws SpecProgramException if an error occurs or is detected during the application of this specification file.
	 */
	public byte[] run(byte[] capability, Map vars) throws SpecParseException, SpecProgramException {
		ByteWalker peek = new ByteWalker(capability);
		int        magic = peek.readShort();

		/* MZ header, indicating a DLL */
		if (magic == 0x5a4d) {
			return runDll(capability, vars);
		}
		/* x64 and x86 Machine values from COFF */
		else if (magic == 0x8664 || magic == 0x14c) {
			return runObject(capability, vars);
		}
		/* we don't know what it is, punt */
		else {
			throw new RuntimeException("Argument is not a COFF or DLL.");
		}
	}

	/**
	 * Apply this specification file's directives to the passed in parameters.
	 *
	 * @param dll_arg our Win32 DLL argument, installed into the environment as {@code $DLL}. This value is lightly checked for validity
	 * 	(e.g., MZ/PE headers exist). The architecture of this DLL determines which target to call from the specification file.
	 * @param vars a mapping of {@code $KEY} to {@code byte[]} values made available during the application of the specification file.
	 * 	Each key should begin with '$' to be accessible within the spec environment.
	 *
	 * @return the position-independent code (usually) generated by applying this specification to the arguments
	 *
	 * @throws SpecParseException if a specification file called {@code run "file.spec"} from this specification has syntax errors.
	 * @throws SpecProgramException if an error occurs or is detected during the application of this specification file.
	 */
	public byte[] runDll(byte[] dll_arg, Map vars) throws SpecParseException, SpecProgramException {
		HashMap env = new HashMap();
		if (vars != null)
			env.putAll(vars);

		env.put("$DLL", dll_arg);

		/* reset all of the error/state information */
		program.reset();

		return program.runDll(env);
	}

	/**
	 * Apply this specification file's directives to the passed in parameters.
	 *
	 * @param object_arg our Win32 COFF argument, installed into the environment as {@code $OBJECT}. This value is lightly checked for validity
	 * 	(e.g., valid Machine header value). The architecture of this object determines which target (x86.o, x64.o) to call from the specification file.
	 * @param vars a mapping of {@code $KEY} to {@code byte[]} values made available during the application of the specification file.
	 * 	Each key should begin with '$' to be accessible within the spec environment.
	 *
	 * @return the position-independent code (usually) generated by applying this specification to the arguments
	 *
	 * @throws SpecParseException if a specification file called {@code run "file.spec"} from this specification has syntax errors.
	 * @throws SpecProgramException if an error occurs or is detected during the application of this specification file.
	 */
	public byte[] runObject(byte[] object_arg, Map vars) throws SpecParseException, SpecProgramException {
		HashMap env = new HashMap();
		if (vars != null)
			env.putAll(vars);

		env.put("$OBJECT", object_arg);

		/* reset all of the error/state information */
		program.reset();

		return program.runObject(env);
	}

	/**
	 * Act on this specification file's directives.
	 *
	 * This API differs in purpose from {@link run}, {@link runDll}, and {@link runObject}. Its purpose is to assemble a PIC program
	 * from known parts, rather than applying PIC to a user-specified capability implemented as a DLL or COFF.
	 *
	 * @param arch is the target architecture label to act on (e.g., x86 or x64)
	 * @param vars a mapping of {@code $KEY} to {@code byte[]} values made available during the application of the specification file.
	 * 	Each key should begin with '$' to be accessible within the spec environment.
	 *
	 * @return the position-independent code generated by executing this specification file.
	 *
	 * @throws SpecParseException if a specification file called {@code run "file.spec"} from this specification has syntax errors.
	 * @throws SpecProgramException if an error occurs or is detected during the application of this specification file.
	 */
	public byte[] buildPic(String arch, Map vars) throws SpecParseException, SpecProgramException {
		HashMap env = new HashMap();
		if (vars != null)
			env.putAll(vars);

		return program.run(arch, arch, env);
	}

	/**
	 * Parse a .spec file and return a ready-to-run {@code LinkSpec} object.
	 *
	 * @param parent the parent file of this .spec content. Returned by {@code #getFile}.
	 * @param content the .spec file content
	 *
	 * @return a LinkSpec object.
	 *
	 * @throws SpecParseException if a syntax error or mis-used command is detected
	 */
	public static LinkSpec Parse(String parent, String content) throws SpecParseException {
		SpecParser parser = new SpecParser();
		parser.parse(content, parent);

		if (parser.getErrors().size() > 0) {
			throw new SpecParseException(parser, parent);
		}

		return parser.getSpec();
	}

	/**
	 * Parse a .spec file and return a ready-to-run {@code LinkSpec} object.
	 *
	 * @param parent The name of the .spec file to read and parse.
	 *
	 * @return a LinkSpec object
	 *
	 * @throws SpecParseException if a syntax error or mis-used command is detected
	 * @throws IOException if we can't read the .spec file
	 */
	public static LinkSpec Parse(String parent) throws SpecParseException, IOException {
		return Parse( parent, CrystalUtils.readStringFromFile(parent) );
	}

	/**
	 * The entrypoint to Crystal Palace. This is our main function, does the stuff you know and love.
	 * @param args the command-line arguments
	 */
	public static void main(String args[]) {
		/* check that we have a verb! */
		if (args.length == 0) {
			CrystalUtils.print_error("Please use ./link or ./piclink");
			return;
		}

		/* check our verb, make sure it's piclink or link */
		String command = args[0];
		if ("run".equals(command) || "buildPic".equals(command)) {
		}
		else {
			CrystalUtils.print_error("Unrecognized verb '" + command + "'");
			return;
		}

		/* check our number of arguments */
		if (args.length < 4) {
			if ("run".equals(command))
				CrystalUtils.print_error("./link [/path/to/loader.spec] [/path/to/file.dll|file.o] [out.bin] <A=...> <B=...>\n\tApply the specified .spec file to build a PIC DLL or COFF loader for file\n\n\tYou may also specify (optional) VARIABLES on the command-line too.\n\n\tFor example:\n\n\tA=04030201 places { 0x04, 0x03, 0x02, 0x01 } into $A.\n\n\tTake care with the native byte order when specifying ints/longs/pointers");
			else if ("buildPic".equals(command))
				CrystalUtils.print_error("./piclink [/path/to/loader.spec] [x86|x64] [out.bin] <A=...> <B=...>\n\tRun the specified .spec file to assemble a PIC program\n\n\tYou may also specify (optional) VARIABLES on the command-line too.\n\n\tFor example:\n\n\tA=04030201 places { 0x04, 0x03, 0x02, 0x01 } into $A.\n\n\tTake care with the native byte order when specifying ints/longs/pointers");
			return;
		}

		/* create a Map of our arguments */
		Map env   = new HashMap();
		for (int x = 4; x < args.length; x++) {
			String split[] = args[x].split("=");
			if (split.length != 2) {
				CrystalUtils.print_error("Argument " + args[x] + " is not KEY=######## format.");
				return;
			}

			if (split[0].startsWith("$")) {
				CrystalUtils.print_error("Do not specify $ for " + split[0] + ". I'll add '$' for you");
				return;
			}

			try {
				byte[] temp = CrystalUtils.hexToBytes(split[1]);
				env.put("$" + split[0], temp);
			}
			catch (Exception ex) {
				CrystalUtils.print_error("Could not convert " + split[1] + " to byte[]: " + ex.getMessage());
				return;

			}
		}

		/* run the tool */
		try {
			if ("run".equals(command)) {
				/* process our arguments */
				byte[] file   = CrystalUtils.readFromFile(args[2]);

				/* load the spec and check for errors */
				LinkSpec spec = LinkSpec.Parse(args[1]);

				/* process the spec */
				CrystalUtils.writeToFile(args[3], spec.run(file, env) );
			}
			else if ("buildPic".equals(command)) {
				/* process our arguments */
				String arch   = args[2];

				/* load the spec and check for errors */
				LinkSpec spec = LinkSpec.Parse(args[1]);

				/* process the spec */
				CrystalUtils.writeToFile(args[3], spec.buildPic(arch, env) );
			}
		}
		catch (SpecParseException specex) {
			CrystalUtils.print_error(specex.toString());
		}
		catch (SpecProgramException progex) {
			CrystalUtils.print_error(progex.toString());
		}
		catch (IOException ex) {
			CrystalUtils.reportException(ex);
		}
	}
}
