package crystalpalace.btf;

import crystalpalace.export.DFR;
import crystalpalace.coff.*;
import crystalpalace.merge.*;
import crystalpalace.util.*;
import java.util.*;

public class Modify {
	protected COFFObject object;

	public Modify(COFFObject object) {
		this.object = object;
	}

	public COFFObject fixX86(String retaddr) {
		if (retaddr == null)
			return object;

		/* let's analyze (disassemble) our code first */
		Code code = Code.Init(object).analyze();

		/* get a mapping of function -> disassembled instructions */
		Map funcs = code.getCodeByFunction();

		return new Rebuilder(code, funcs).rebuild(new FixX86References(code, retaddr));
	}

	public COFFObject fixBSS(String getbss) {
		if (getbss == null)
			return object;

		/* let's analyze (disassemble) our code first */
		Code code = Code.Init(object).analyze();

		/* get a mapping of function -> disassembled instructions */
		Map funcs = code.getCodeByFunction();

		if ( "x64".equals(object.getMachine()) )
			return new Rebuilder(code, funcs).rebuild(new FixBSSReferencesX64(code, getbss));
		else
			return new Rebuilder(code, funcs).rebuild(new FixBSSReferencesX86(code, getbss));
	}

	public COFFObject resolve(DFR resolvers) {
		/* No resolver function? Well then... don't do anything */
		if (!resolvers.hasResolvers())
			return object;

		/* let's analyze (disassemble) our code first */
		Code code = Code.Init(object).analyze();

		/* get a mapping of function -> disassembled instructions */
		Map funcs = code.getCodeByFunction();

		return new Rebuilder(code, funcs).rebuild(new ResolveAPI(code, resolvers));
	}

	public COFFObject modify(boolean preserveFirst, Set options) {
		/* If we're not modifying anything, then let's not do anything */
		if (options.size() == 0)
			return object;

		/* let's analyze (disassemble) our code first */
		Code code = Code.Init(object).analyze();

		/* get a mapping of function -> disassembled instructions */
		Map funcs = code.getCodeByFunction();

		/* make the go() entry point function the first one in our program */
		if (options.contains("+gofirst"))
			funcs = new GoFirst(code).apply(funcs);

		/* apply the LTO if that option was selected */
		if (options.contains("+optimize"))
			funcs = new LinkTimeOptimizer(code).apply(funcs);

		/* apply the function disco, if this option was selected */
		if (options.contains("+disco"))
			funcs = new FunctionDisco(code).apply(preserveFirst, funcs);

		/* rebuild the program */
		if (options.contains("+mutate")) {
			return new Rebuilder(code, funcs).rebuild(new Mutator(code));
		}
		else {
			return new Rebuilder(code, funcs).rebuild();
		}
	}

	public static void main(String args[]) {
		if (args.length == 0) {
			CrystalUtils.print_error("./disassemble <+mutate,+optimize,+disco,+gofirst> [/path/to/file.o]");
			return;
		}

		COFFObject obj  = null;
		String     opts = "";

		try {
			if (args.length == 1) {
				obj  = new COFFParser().parse(CrystalUtils.readFromFile(args[0])).getObject();
			}
			else if (args.length == 2) {
				opts = args[0];
				obj  = new COFFParser().parse(CrystalUtils.readFromFile(args[1])).getObject();
			}

			/* normalize our COFF */
			COFFMerge merge = new COFFMerge();
			merge.merge(obj);
			merge.finish();
			obj = merge.getObject();

			obj = new Modify(obj).modify(false, "".equals(opts) ? new HashSet() : CrystalUtils.toSet(opts));

			CodeUtils.print(Code.Init(obj).analyze());
		}
		catch (Exception ex) {
			CrystalUtils.handleException(ex);
		}
	}
}
